---------------------------------------------------------------------------------------------------
-- WP Converter - by Edward Mendelson 26 October 2013, http://www.wpdos.org
---------------------------------------------------------------------------------------------------
-- Filename checking and output-naming based on an earlier script by Smokey Ardisson
---------------------------------------------------------------------------------------------------
(*
This application uses multiple multiple methods to export WordPerfect documents into 
RTF, DOC, or PDF formats. It also exports WordPerfect for the Mac files into WPDOS 5.1
format for use in Windows applications. It converts documents created by all versions 
of WordPerfect for DOS, Windows, or the Macintosh.

By default, the application converts WP documents into RTF format. If you change the 
name of the application so that it includes DOC or PDF (for example "WP Converter
PDF"), it will export DOC or PDF files instead.

For most conversions, the application uses the WP ConvertPerfect utility, running under
theDOSBox DOS emulator. ConvertPerfect exports files in RTF format or the DOC format
used by Word for DOS 6.0 (which some versions of Microsoft Word may refuse to open
for security reasons). PDF files are created by using an internal OS X utility to convert
RTF files exported by ConvertPerfect.

For RTF, DOC, and PDF conversions of files in the format used by WordPerfect 3.5
Enhanced for the Mac, the application uses a copy of WPMac 3.5 Enhanced running in
the SheepShaver emulator. WPMac 3.5 Enhanced is also used for PDF exports from all
WPMac files, because WPMac may produce more accurate PDF output than exports from
the DOS-based utility.You can prevent the application from using WPMac by modifying
the name of the application so that it includes the string "NoMac". (The "NoMac" option,
when used with WPMac 3.5 Enhanced documents, produces minimally-formatted files
instead of correctly-formatted ones.)

The application works by copying your original file, then using a conversion filter to
export a converted version of the copied file. The application then copies the converted
version back to the same folder with the original, but with an additional .RTF, .DOC,
or .PDF extension added to the original filename.  If an existing converted copy already
exists, the new copy will havea number at the end of the filename, before the
extension.

Warning: This application may respond unpredictably or lock up if you try to convert
password-protected documents or WPMac documents that contain macros. Open such
documents in a separate copy of WordPerfect and remove any passwords or macros.

An additional feature exports WPMac files to WPDOS 5.1 format, which can be opened by
many Windows applications. Files exported to WPDOS 5.1 format include working
cross-references and other features that are not exported to other formats. To use this
feature, modify the the name of the application so that it includes the string
"ToWP51". The exported files will have the extension .WP5.

Alternatively, you my export WPMac files to WPDOS 6.x format by adding "ToWP60" to
the name of the application.The exported files will have the extension .WP6. Or you
may export WPMac files to the more widely-recognized WPMac 3.0 format, by adding
"ToWP3" the name of the application; the exported files will have the extension .WP3.

Troubleshooting: A separate AppleScript runs in the SheepShaver emulator to control
the operations of WPMac. You may edit this AppleScript by adding "NoStart" and "BII" to
the name of the WP Converter application. "NoStart" prevents the SheepShaver-based
AppleScript from performing the conversion and "BII" causes the BasiliskII emulator to
be used instead of SheepShaver, because the AppleScript Editor application does not run
under SheepShaver.

Other options accessible by changing the name of the application include "NoQuit",
which prevents SheepShaver (or BasiliskII) from closing after performing the conversion.

A note on files created by WordPerfect for the Mac 3.5 Enhanced: When the "NoMac"
option is selected, and ONLY when the "NoMac" option is selected, this application uses
the open-source wpd2odt utility to convert the original WPMac 3.5 Enhanced file to
ODT format, then uses the OS X textutil utility to convert the ODT file to RTF format. 
("DOC" files created by this option are merely the exported RTF files saved with a .DOC
extension; such files can be opened by all applications that can open standard DOC files.)
Remember that these limitations ONLY occur when "NoMac" is selected, and they only
apply to files created in the format used by WPMac 3.5Enhanced (3.5e).
*)
----------------------------------------------------------------------------------------------------
-- user-modifiable variable: use "rtf" or "doc" or "pdf" to change the default output

property defaultOutput : {"rtf" as Unicode text}

-- do NOT try to change anything below this line unless you know what you are doing!
----------------------------------------------------------------------------------------------------

property parent : AppleScript

property typeList : {"WPD0", "WPD1", "WPD2", "WPD3", "WPD4", "WP42", "WP50", ".WP5", ".WP6", "WPPC", "sPD1", "sPD2", "sPD3"}
property extensionList : {"wpd", "wp", "wp5", "wp6", "WPD", "WP", "wpt"} --needs to exclude the period!
property macTypeList : {"WPD0", "WPD1", "WPD2", "WPD3", "WPD4", "sPD1", "sPD2", "sPD3"} --filetypes for WPMac documents
property creatorTypeList : {"WPC2", "SSIW"} -- not used for anything just yet!
-- the following are here to make sure they exist when needed, but we have to zero them in the on run/on open handlers because the script stores the values
property badExtensionList : {"DMG", "PDF", "XLSX", "DOCX", "XLS", "XLSX", "ZIP"}
property rejectedList : {""} -- list of files that get rejected by type/extension check
property errorList : {""} -- list of files that we encounter errors on
property reportRejects : {""} -- text + list of rejected files to write when report requested
property reportErrors : {""} -- text + list of error-causing files to write when report requested
property time_stamp : {""} -- time of writing report

property useMac : false -- changed by the input format and other options
property useBII : false -- use BasiliskII for debugging (the script editor only works under BII)
property wpmacOne : false -- changed if WPMac1.x file detected
property wpmacFour : false -- changed if WPMac3.5e file detected
property typeWP42 : false -- changed if Mac file with filetype WP42 detected
property batch : "no" -- changed to "yes" if multiple items dropped
property theFollowing : "This file was " -- these three strings change for multiple items
property notValid : "This file may not be a valid WordPerfect document."
property continueConv : ""
-- property alertId : {""}
property sheepPath : ""
property macPath : ""
property wppcDoc : false
property outString : ""
property shortName : ""
property outUpper : ""
property macType : false -- changed if input file created by WPMac
property firstPara : ""

-- all the following can be set by changing the application name
property doPDF : false
property doRTF : false
property doDOC : false
property doWP3 : false
property doWP5 : false
property doWP6 : false
property noStart : false -- doesn't start conversion script in WPMac
property noQuit : false -- doesn't exit WPMac after running conversion script
property allMac : false -- use WPMac for all conversions
property noMac : false -- never use WPMac for conversions

property reqVers : 4
global sysvers

on run -- handles double-click on the app
	versionCheck()
	setVars()
	setupVM()
	getMyName()
	-- on run we select one file
	activate
	set selectFile to (choose file with prompt "Select a WordPerfect file to convert to " & toUpper(outString as Unicode text) & " format:")
	
	if outString is {"pdf" as Unicode text} then set doPDF to true
	-- else
	--	set doPDF to false
	-- end if
	
	my checkValid(selectFile, batch, doPDF, outString)
	-- displays a warning at end about non-converted file if selected file fails type/extension check
	
	if rejectedList  "" then
		tell application "Finder"
			activate
			display dialog "The file you selected " & return & rejectedList & return & return & "was not converted.  " & notValid buttons {"OK"} default button 1 with icon stop
		end tell
	end if
end run

-- do when item dropped
on open inItemList
	versionCheck()
	setVars()
	setupVM()
	getMyName()
	
	-- set up for messages about multiple files when files dropped
	if (inItemList's length) is greater than 1 then
		set batch to "yes"
		set theFollowing to "The following files were "
		set notValid to "These files may not be valid WordPerfect documents."
		set continueConv to " Conversion of other files will continue."
	end if
	
	repeat with theItem in inItemList -- recursion of folders is buggy; only active in dev script
		my checkValid(theItem, batch, doPDF, outString)
	end repeat
	
	-- displays list of non-converted (rejected or error-encountering files and eventually offers to save a list of them	
	if rejectedList  "" or errorList  "" then
		tell application "Finder"
			activate
			-- Some files were not converted or experienced errors during conversion.  They may not be valid WordPerfect documents.
			display dialog theFollowing & "not converted:" & return & rejectedList & errorList & return & return & notValid buttons {"Save Report to Desktop", "Quit"} default button 2 with icon caution
		end tell
		if the button returned of the result is "Save Report to Desktop" then
			-- set up our final output (only if we're actually going to write it out)
			if rejectedList  "" then set reportRejects to ("No Conversion Attempted" & return & "-----------------------" & rejectedList)
			if errorList  "" then set reportErrors to ("Errors During Conversion" & return & "------------------------" & errorList)
			
			-- display dialog reportRejects
			try
				my timeStamp()
				set theReportPath to ((path to desktop) as text) & "WPD Converter Error Report " & time_stamp & ".txt"
				set theReport to (open for access file theReportPath with write permission)
				set eof theReport to 0 --if you want to overwrite rather than append
				
				if rejectedList = "" then write (reportErrors) to theReport
				if errorList = "" then write (reportRejects) to theReport
				if errorList  "" and rejectedList  "" then write (reportRejects & return & return & reportErrors) to theReport
				close access theReport
			on error the writing_error number the writing_error_num
				set the writing_error_text to "Error " & the writing_error_num & ": " & the writing_error
				activate --makes sure the dialog is not hidden
				display dialog "There was a problem saving the report." & return & return & the writing_error_text & return & return & "Sorry." buttons {"Quit"} default button 1 with icon stop
				return the writing_error_text
			end try
		end if
	end if
end open

on GetFileInfo(cnvFile, useMac, doPDF, outString) -- gets various parts of the filename in order to test for existing files with that name
	set theCounter to 0 -- initializes our unique filename counter
	tell application "Finder" -- Finder has to get that
		set the parent_container_path to ((the container of cnvFile) as text)
		if the last character of the parent_container_path is ":" then -- fix for Desktop
		else
			set the parent_container_path to the parent_container_path & ":"
		end if
		
	end tell
	
	-- setup for exporting WPMac 3.5e or 1.x to 3.x format
	if doWP3 is true then
		if ((wpmacFour is true) or (wpmacOne is true)) then
			set outString to {"wp3" as Unicode text}
		end if
	end if
	
	-- setup for exporting WPMac files to WPDOS 5.1 format
	if doWP5 is true then
		if macType is true then
			set outString to {"wp5" as Unicode text}
		end if
	end if
	
	-- setup for exporting WPMac files to WPDOS 6.x format
	if doWP6 is true then
		if macType is true then
			set outString to {"wp6" as Unicode text}
		end if
	end if
	
	-- display dialog outString
	set theExtension to ("." & outString)
	copy theExtension to theAddition
	--pass all these bits of data to the checking routine, so it can test for a duplicate filename, increment the counter to make unique, recontstruct the filename and test again, and then hand off the the actual conversion!
	my checkPath(parent_container_path, cnvFile, theAddition, theExtension, theCounter, useMac, doPDF, outString)
end GetFileInfo

on checkPath(parent_container_path, cnvFile, theAddition, theExtension, theCounter, useMac, doPDF, outString)
	--display dialog theCounter -- (name of cnvFile) & " : " & theCounter
	tell application "Finder" -- only Finder can "exists" so this is all a Finder tell block :(
		-- can't figure out how to pull the rest out of the Finder block; it works in simple scripts but not here
		-- the Finder might also be needed in subsequent steps here
		if exists item (parent_container_path & the name of cnvFile & theAddition) then
			--increment our unique counter and the "filename extension"
			set theCounter to (theCounter + 1)
			set theAddition to (" " & theCounter & theExtension) --used to be "-"
			-- try again for a unique filename using the updated values; pass all the fragments back to ourselves, and keep trying until we get a unique name
			my checkPath(parent_container_path, cnvFile, theAddition, theExtension, theCounter, useMac, doPDF, outString)
		else
			set newPath to (quoted form of (POSIX path of (cnvFile as alias) & theAddition))
			my wpd2rtf(cnvFile, newPath, theExtension, batch, useMac, doPDF, outString) -- we should have a clean filename now; pass the original file, and the converted file's full name, and batch or single files, to the conversion routine
		end if
	end tell
end checkPath

on wpd2rtf(cnvFile, newPath, theExtension, batch, useMac, doPDF, outString) --finally, we do the actual conversion!
	-- display dialog "wpmacOne :" & wpmacOne & " - wpmacFour: " & wpmacFour
	set thisApp to (path to me) as text
	set dosDirPath to (thisApp & "Contents:Resources:Files") as text
	set dosDirPosix to (POSIX path of (dosDirPath as alias))
	set tempPath to (thisApp & "Contents:Resources:Files:Drive:Temp") as text
	set tempPosix to (POSIX path of (tempPath as alias)) -- tempPosix ends with "/"
	set infilePosix to quoted form of (tempPosix & "infile")
	set cnvPosix to quoted form of (POSIX path of (cnvFile as alias))
	
	set wfwTempPath to (thisApp & "Contents:Resources:Files:Drive:WFW:TempWFW") as text
	set wfwTempPosix to quoted form of (POSIX path of (wfwTempPath as alias))
	
	set odtApp to (thisApp & "Contents:Resources:Files:wpd2odt") as text
	set odtPosix to quoted form of (POSIX path of (odtApp as alias))
	
	-- for WPMac-based conversion
	runOK()
	set psOutput to quoted form of (tempPosix & "outfile.ps")
	set outfilePosix to quoted form of (tempPosix & "outfile" & theExtension)
	set write35doRTF to quoted form of (tempPosix & "35doRTF")
	set write35PrintPDF to quoted form of (tempPosix & "35PrintPDF")
	set write35export30 to quoted form of (tempPosix & "35export30")
	set write35export51 to quoted form of (tempPosix & "35export51")
	set write35export60 to quoted form of (tempPosix & "35export60")
	set bIIPath to (thisApp & "Contents:Resources:Files:BasiliskII.app:Contents:MacOS") as text
	set bIIPosix to quoted form of (POSIX path of (bIIPath as alias))
	set writeNoStart to quoted form of (tempPosix & "DoNotStart")
	set writeNoQuit to quoted form of (tempPosix & "DoNotQuit")
	-- end for WPMac-based variables
	
	tell application "Finder"
		set cnvName to the name of cnvFile
	end tell
	
	-- delete old input file
	try
		do shell script "rm " & infilePosix
	end try
	try
		do shell script "cp " & cnvPosix & space & infilePosix
	on error
		display dialog "Unknown error." & return & return & "Could not copy the selected file to the conversion directory." & return & return & "Quitting." buttons {"OK"} default button 1 with icon caution
		error number -128
	end try
	try
		set shortName to my nameOnly(newPath)
	end try
	
	copy my toUpper(outString as Unicode text) to outUpper
	
	set rtfDone to false
	
	-- refine whether or not to use WPMac
	
	if noMac is false then
		if doPDF is true then
			if macType is true then
				set useMac to true
			end if
		end if
	end if
	-- if (noMac is false) and (doPDF is true) and (macType is true) then set useMac to true
	if noMac is true then set useMac to false
	if allMac is true then set useMac to true
	
	try
		if useMac is true then
			try
				do shell script "rm " & psOutput
			end try
			-- remove old control files	
			try
				do shell script "rm " & write35doRTF
			end try
			try
				do shell script "rm " & write35PrintPDF
			end try
			try
				do shell script "rm " & write35export30
			end try
			try
				do shell script "rm " & write35export51
			end try
			try
				do shell script "rm " & write35export60
			end try
			try
				do shell script "rm " & writeNoStart
			end try
			try
				do shell script "rm " & writeNoQuit
			end try
			
			if outString is {"rtf" as Unicode text} then
				do shell script "echo > " & write35doRTF
			end if
			if outString is {"pdf" as Unicode text} then
				do shell script "echo > " & write35PrintPDF
			end if
			
			if doWP3 is true then
				if ((wpmacFour is true) or (wpmacOne is true)) then
					do shell script "echo > " & write35export30
				end if
			end if
			
			if doWP5 is true then
				if macType is true then
					do shell script "echo > " & write35export51
				end if
			end if
			
			if doWP6 is true then
				if macType is true then
					do shell script "echo > " & write35export60
				end if
			end if
			
			if noStart is true then do shell script "echo > " & writeNoStart
			if noQuit is true then do shell script "echo > " & writeNoQuit
			
			set rtfDone to false
			
			-- run SheepShaver or BasiliskII
			if useBII is false then
				set checkProcess to "SheepShaver"
				try
					do shell script "open -a" & space & quoted form of sheepPath & space & quoted form of macPath
				on error
					display dialog "Couldn't start SheepShaver. Quitting." buttons {"Bummer"} default button 1 with icon stop
					error number -128
				end try
			else if useBII is true then -- use BasiliskII instead of SheepShaver when editing the System 7 Applescript
				set checkProcess to "BasiliskII"
				try
					do shell script "cd " & bIIPosix & " ; ./BasiliskII --config ../biiprefs &> /dev/null &"
				on error
					display dialog "Couldn't start Basilisk II. Quitting." buttons {"Bummer"} default button 1 with icon stop
					error number -128
				end try
			end if
			
			-- wait for the emulator to start
			tell application "System Events"
				repeat until (process checkProcess exists)
					delay 0.1
				end repeat
				delay 0.2
				repeat until ((process checkProcess exists) is false)
					delay 1
				end repeat
			end tell
			delay 0.5
			
			-- use OS X pstopdf to convert PostScript output from WPMac to PDF
			if doPDF is true then
				try
					do shell script "pstopdf " & psOutput & " -o " & outfilePosix
				on error
					display dialog "Error converting PostScript file to PDF." buttons {"Bad news"} default button 1 with icon caution
				end try
			end if
			
			try
				do shell script "/bin/ls " & outfilePosix -- do we have the output file?
				set rtfDone to true
			on error errMsg number errNum
				
				tell application "Finder" -- have Finder give us "name of cnvFile"
					set theErrorFileName to (name of cnvFile) --as string
				end tell
				activate -- makes sure the dialog is not hidden
				-- notifies user which file caused a problem and the error text
				if noStart is true then
					display dialog "The exported output file was not found." & return & return & "This is normal if you used the NoStart option and the WPMac conversion macro did not convert a file while the emulator was open." buttons {"OK"} default button 1 with icon caution
				else
					display dialog "An error occurred while attempting to convert the file: " & theErrorFileName & return & return & the error_text & return & return & continueConv buttons {"OK"} default button 1 with icon caution
					-- end tell -- when the finder was used to display all messages
					set errorList to (errorList & return & (theErrorFileName)) --used to be cnvFile
					return the error_text
				end if
			end try
			
			-- clean up input and control files
			try
				do shell script "rm " & infilePosix
			end try
			try
				do shell script "rm " & write35doRTF
			end try
			try
				do shell script "rm " & write35PrintPDF
			end try
			try
				do shell script "rm " & write35export30
			end try
			try
				do shell script "rm " & write35export51
			end try
			try
				do shell script "rm " & write35export60
			end try
			try
				do shell script "rm " & writeNoStart
			end try
			try
				do shell script "rm " & writeNoQuit
			end try
			try
				do shell script "rm " & psOutput
			end try
			
		else -- if useMac is false then
			if wpmacFour is false then
				if wpmacOne is true then -- for WPMac 1.x only
					do shell script "cd " & quoted form of dosDirPosix & " && " & " ./DOSBox -c " & quote & "set FN=" & cnvName & quote & " -c " & quote & "set OUT=" & shortName & quote & " -c  " & quote & "set BATCH=" & batch & quote & " -c 'mount C ./Drive' -c 'C:' -c 'cd /' -c cls -c  " & quote & "autowfw " & outUpper & quote & " -conf  ./wpcv.conf"
					set rtfDone to true
					try
						do shell script "rm " & wfwTempPosix & "*"
					end try
				else if typeWP42 is true then -- for WP42 files created by exporting from WPMac only
					do shell script "cd " & quoted form of dosDirPosix & " && " & " ./DOSBox -c " & quote & "set FN=" & cnvName & quote & " -c " & quote & "set OUT=" & shortName & quote & " -c  " & quote & "set BATCH=" & batch & quote & " -c 'mount C ./Drive' -c 'C:' -c 'cd /' -c cls -c  " & quote & "autorun4 " & outUpper & quote & " -conf  ./wpcv.conf"
					set rtfDone to true
					try
						do shell script "rm " & wfwTempPosix & "*"
					end try
				else -- for all standard files
					do shell script "cd " & quoted form of dosDirPosix & " && " & " ./DOSBox -c " & quote & "set FN=" & cnvName & quote & " -c " & quote & "set OUT=" & shortName & quote & " -c  " & quote & "set BATCH=" & batch & quote & " -c 'mount C ./Drive' -c 'C:' -c 'cd /' -c cls -c  " & quote & "autorun " & outUpper & quote & " -conf  ./wpcv.conf"
					set rtfDone to true
				end if
				
			else if wpmacFour is true then -- wpmacFour is WPMac 3.5 Enhanced
				
				if sysvers is less than 6 then
					set dialogText to "Under OS X 10.4 and 10.5, the NoMac option cannot be used when converting WPMac 3.5e files." & return & return & "Please try without the NoMac option." & return & return & "Quitting."
					my errorQuit(dialogText)
				end if
				
				my showAlert()
				
				if doPDF is true then
					set theExtension to ".pdf"
				else
					set theExtension to ".rtf"
				end if
				
				try
					do shell script odtPosix & space & cnvPosix & " " & quoted form of dosDirPosix & "Drive/Temp/outfile.odt"
					-- do shell script "cd " & quoted form of dosDirPosix & " && " & odtPosix & space & cnvPosix & "  > " & quoted form of dosDirPosix & "Drive/Temp/outfile.odt"
					set rtfDone to false
				on error the error_message number the error_number -- 'gracefully' handles errors from wpd2odt
					set rtfDone to false
					set the error_text to the error_message --"Error: " & the error_number & ". " & the error_message
					--wpd2odt throws no error numbers and includes the word Error, so dispensing with Error: #.
					tell application "Finder" -- have Finder give us "name of cnvFile"
						set theErrorFileName to (name of newPath) --as string
					end tell
					activate --makes sure the dialog is not hidden
					-- notifies user which file caused a problem and the error text
					display dialog "wpd2odt encountered an error while converting to an intermediate ODT file: " & theErrorFileName & return & return & the error_text & return & return & "This file may still be converted by opening it in a word processor and saving it in another format." & continueConv buttons {"OK"} default button 1 with icon caution
					--end tell -- when the finder used to display all messages
					set errorList to (errorList & return & (theErrorFileName)) --used to be cnvFile
					return the error_text
				end try
				try
					do shell script "cd " & quoted form of dosDirPosix & "Drive/Temp " & "&&" & " textutil -convert rtf outfile.odt -output outfile.rtf && rm outfile.odt"
					set rtfDone to true
				on error the error_message number the error_number -- 'gracefully' handles errors from textutil, maybe
					set rtfDone to false
					set the error_text to "Error: " & the error_number & ". " & the error_message
					-- not sure if textutil throws error messages
					tell application "Finder" -- have Finder give us "name of cnvFile"
						set theErrorFileName to (name of newPath) --as string
					end tell
					activate -- makes sure the dialog is not hidden
					-- notifies user which file caused a problem and the error text
					display dialog "textutil encountered an error while converting the intermediate ODT file to rtf format: " & theErrorFileName & return & return & the error_text & return & return & "This file may still be converted by opening it in a word processor and saving it in another format." & continueConv buttons {"OK"} default button 1 with icon caution
					-- end tell -- when the finder used to display all messages
					set errorList to (errorList & return & (theErrorFileName)) --used to be cnvFile
					return the error_text
				end try
				
				my closeAlert()
			else
				activate -- makes sure the dialog is not hidden
				display dialog "Something went very wrong in attemping to convert a file." buttons {"Ouch"} default button 1 with icon caution
				error number -128
			end if
			
		end if
	end try
	
	if rtfDone is true then
		-- display dialog doPDF
		if doPDF is true then
			try
				do shell script "cupsfilter -D " & quoted form of (tempPosix & "OUTFILE.RTF") & " > " & quoted form of (tempPosix & "outfile.pdf")
			on error errMsg
				activate
				display dialog errMsg
				try
					do shell script "cupsfilter -D " & quoted form of (tempPosix & "outfile.rtf") & " > " & quoted form of (tempPosix & "outfile.pdf")
				on error errMsg
					display dialog errMsg
				end try
			end try
		end if
		
		set outfilePosix to quoted form of (tempPosix & "outfile" & theExtension)
		-- display dialog outfilePosix
		try
			do shell script "cp " & outfilePosix & space & newPath
		on error
			activate -- makes sure the dialog is not hidden
			display dialog "There was a problem copying the converted file " & shortName & " to the same folder with the original file. Please try again with the same original file." buttons {"OK"} default button 1 with icon caution
		end try
		try
			do shell script "rm " & outfilePosix
			do shell script "rm " & infilePosix
		end try
	end if
end wpd2rtf

on checkValid(checkItem, batch, doPDF, outString)
	-- display dialog "start checkValid outstring: " & outString
	set isWPDoc to false
	set wpmacOne to false
	set typeWP42 to false
	set wpmacFour to false
	tell application "System Events"
		set isFilePackage to package folder of checkItem
		set checkExtension to the name extension of checkItem
		set checkName to the name of checkItem
	end tell
	-- display dialog checkExtension
	if isFilePackage is true then
		tell application "System Events" to set packageName to the name of checkItem
		-- display dialog packageName & " is not a conventional file." & return & return & "I cannot convert it." buttons {"OK"} with icon note
		set rejectedList to rejectedList & return & packageName
	else if checkExtension is in the badExtensionList then
		set rejectedList to rejectedList & return & checkName
	else
		tell application "Finder"
			-- test for WPMac1.x and WPMac3.5e
			
			if the name extension of checkItem is in the extensionList or the file type of checkItem is in the typeList then
				set isWPDoc to true
				set useMac to false
			else
				-- test for binary signature of WPC document files
				set isWPDoc to ((read file (checkItem as text) from 1 to 4 as data) is data rdatFF575043)
				set useMac to false
			end if
			
			if the file type of checkItem is in the macTypeList then
				set macType to true
			else
				set macType to false
			end if
			
			if the file type of checkItem is "WPD4" then
				set useMac to true
				set wpmacFour to true
				set isWPDoc to true
			end if
			
			if the file type of checkItem is "WPD3" then
				if doWP5 is true or doWP6 is true then
					set useMac to true
				end if
			end if
			
			if the file type of checkItem is "WPD2" then
				if doWP5 is true or doWP6 is true then
					set useMac to true
				end if
			end if
			
			if the file type of checkItem is "WPD1" then
				if doWP5 is true or doWP6 is true then
					set useMac to true
				end if
			end if
			
			if the file type of checkItem is "WPD0" then
				-- set useMac to false
				if doWP3 is true then
					set useMac to true
				end if
				if doWP5 is true or doWP6 is true then
					set useMac to true
				end if
				set wpmacOne to true
				set isWPDoc to true
			end if
			
			if the file type of checkItem is "WP42" then
				-- set useMac to false
				set typeWP42 to true
				set isWPDoc to true
			end if
			
			if the creator type of checkItem is "SSIW" then
				-- set useMac to false
				set typeWP42 to true
				set isWPDoc to true
			end if
			
			if isWPDoc is true then
				if (typeWP42 is false) and (wpmacOne is false) then
					-- set wpBytes to false
					set isReadable to true
					-- set cryptWP to false
					-- set warnEncrypt to false
					set fRef to (open for access file (checkItem as text)) -- Open the file just once and use the returned access reference for the three reads.
					try
						set isReadable to (((read fRef for 4 as data) is data rdatFF575043) and ((read fRef from 13 for 1 as data) is data rdat00))
						-- display dialog isReadable
					end try
					close access fRef -- Close the file again.
					if isReadable is false then
						set isWPDoc to false
						display dialog the name of checkItem & " seems to be an encrypted WordPerfect file." & return & return & "I cannot convert this file." buttons {"OK"} with icon note
						set rejectedList to rejectedList & return & the name of checkItem
					end if
				end if
			end if
			
			if isWPDoc is false then
				if the file type of checkItem is missing value and the creator type of checkItem is missing value then
					set firstPara to ""
					set firstPara to my readFile(checkItem)
					activate -- makes sure the dialog is not hidden
					set whatFile to display dialog "I cannot determine the file type of " & the name of checkItem & "." & return & return & "The first paragraph of the file (may include codes) is: " & return & return & "-------------------------------------------------" & return & firstPara & return & "-------------------------------------------------" & return & return & "If you believe that this file is a WordPerfect document, choose the option that seems most likely:" buttons {"WPMac 1.x file", "WPDOS 4.x file", "Not a WP file"} default button 3 with icon note
					if button returned of whatFile is "WPMac 1.x file" then
						set isWPDoc to true
						set wpmacOne to true
					else if button returned of whatFile is "WPDOS 4.x file" then
						set isWPDoc to true
						set typeWP42 to true
					else if button returned of whatFile is "Not a WP file" then
						set isWPDoc to false
					end if
				end if
			end if
			
			if isWPDoc is true then
				my GetFileInfo(checkItem, useMac, doPDF, outString)
				-- else statement to generate list of invalid files 
			else if isWPDoc is false then
				set rejectedList to rejectedList & return & the name of checkItem
			end if
		end tell
	end if
end checkValid

on timeStamp()
	set time_stamp to do shell script "date +%Y-%m-%d\\ %H:%M:%S"
	set oldTID to AppleScript's text item delimiters
	set AppleScript's text item delimiters to ":"
	set theText to time_stamp's text items
	set AppleScript's text item delimiters to "-"
	set time_stamp to theText as text
	set AppleScript's text item delimiters to oldTID
end timeStamp

on versionCheck()
	-- Running on Tiger or later?
	set sysvers to ((system attribute "sysv") mod 4096 div 16)
	--if ((system attribute "sysv") mod 4096 div 16) < reqVers then
	if sysvers < reqVers then
		set dialogText to "This application runs under OS X 10.4 or later, and you seem to have an earlier version of OS X."
		errorQuit(dialogText)
	end if
	
	-- Running on an Intel Mac?
	set checkPlatform to 0
	set checkPlatform to system attribute "sysa"
	if checkPlatform = 2 then
		set dialogText to "This system runs only on Intel Macs, and you seem to have a PowerPC machine."
		errorQuit(dialogText)
	end if
	(*
	try
		do shell script "which textutil" -- is textutil available on this system?
	on error
		activate -- makes sure the dialog is not hidden
		display dialog "This application requires OS X 10.4 or higher." buttons {"OK"} with icon stop giving up after 30
		error number -128
	end try
    *)
end versionCheck

on getMyName()
	set doPDF to false
	set doRTF to false
	set doDOC to false
	
	-- following sets outString to {"rtf" as Unicode text}
	copy defaultOutput to outString
	-- display dialog "after copy default to outstring outstring: " & outString
	
	tell application "System Events" to set myName to name of (get properties of (path to me))
	set myName to text 1 thru -5 of myName
	
	if myName contains "pdf" then
		set outString to {"pdf" as Unicode text}
		set doPDF to true
	end if
	
	try
		if myName contains "rtf" then
			set doRTF to true
			if (doPDF) then error
			set outString to {"rtf" as Unicode text}
		end if
		if myName contains "doc" then
			set doDOC to true
			if (doPDF) then error
			set outString to {"doc" as Unicode text}
		end if
		if doDOC and doRTF then error
	on error
		activate -- makes sure the dialog is not hidden
		display dialog "My name may contain one (only) of these strings: pdf, doc, rtf." & return & return & "If none of these three are part of my name, I will default to exporting files in RTF format." buttons {"If you say so"} default button 1 with icon caution
		error number -128
	end try
	
	-- for useMac
	try
		if myName contains "ToWP3" then set doWP3 to true
		if myName contains "ToWP51" then
			if (doWP3) then error
			set doWP5 to true
		end if
		if myName contains "ToWP60" then
			if (doWP3) then error
			if (doWP5) then error
			set doWP6 to true
		end if
		
	on error
		activate -- makes sure the dialog is not hidden
		display dialog "My name may contain one (only) of strings ToWP3, ToWP51, or ToWP60." & return & return & "These options apply only to WPMac files, and cause them to be exported to WPMac 3.x, WPDOS 5.1, WPDOS 6.x formats." buttons {"If you say so"} default button 1 with icon caution
		error number -128
	end try
	
	if myName contains "NoStart" then set noStart to true
	if myName contains "NoQuit" then set noQuit to true
	if myName contains "BII" then set useBII to true
	if myName contains "NoMac" then set noMac to true
	if myName contains "AllMac" then set allMac to true
	if noMac and (doWP3 or doWP5 or doWP6) then
		activate -- makes sure the dialog is not hidden
		display dialog "Please do not include both NoMac and either ToWP51, ToWP60, or ToWP3 in the same name." & return & return & "These options conflict with each other." buttons {"If you say so"} default button 1 with icon caution
		error number -128
	end if
	-- end for useMac
	
	if myName contains "Help" then my getHelp()
	
	-- set defaultAction to not ((doPDF) or (doDoc) or (doRTF))	
end getMyName

on toUpper(theString)
	return (do shell script "echo " & quoted form of theString & " | tr \"[:lower:]\" \"[:upper:]\"")
end toUpper

on nameOnly(newPath)
	set {TID, text item delimiters} to {text item delimiters, "/"}
	set shortName to last text item of newPath as text
	set TID to AppleScript's text item delimiters
	if (the last character of shortName is (ASCII character 39)) then set shortName to text 1 thru -2 of shortName as text
end nameOnly

on showAlert()
	if wpmacFour is true then
		do shell script "osascript -e 'tell application \"Automator Runner\"  to activate'"
		do shell script "osascript -e 'tell application \"Automator Runner\"  to display dialog \"Large files may take a long time to convert.\" & return & return & \"When files in WPMac 3.5 Enhanced format are converted, this message is the only visible indication.\" buttons {\"Conversion in process\"} with title \"WP Converter\" giving up after 60' &> /dev/null &"
	else
		do shell script "osascript -e 'tell application \"Automator Runner\"  to activate'"
		do shell script "osascript -e 'tell application \"Automator Runner\"  to display dialog \"Large files may take a long time to convert.\" & return & return & \"You may minimize the DOSBox window, but do NOT try to close it, even if it seems inactive.\" buttons {\"Conversion in process\"} with title \"WP Converter\" giving up after 60' &> /dev/null &"
	end if
	(*
	set alertId to ""
	if wpmacFour is true then
		set alertFile to ((path to me as Unicode text) & "Contents:Resources:Files:WPMac35eAlert.rtf")
	else
		set alertFile to ((path to me as Unicode text) & "Contents:Resources:Files:ConvertAlert.rtf")
	end if
	tell application "TextEdit"
		open file alertFile
		set bounds of front window to {20, 40, 345, 240}
		set alertId to item 1 of every window
		activate
	end tell
	return alertId
*)
end showAlert

on closeAlert()
	set asProcess to quoted form of "Automator Runner"
	try
		do shell script "killAll " & asProcess
	end try
	(*
	try
		tell application "TextEdit" to close alertId
		set countWindows to the count (the windows of application "TextEdit")
		if countWindows is 0 then
			tell application "TextEdit" to quit
		end if
	end try
*)
end closeAlert

on setupVM()
	-- Set up paths in this app
	set sheepApp to (path to me) & "Contents:Resources:Files:SheepShaver.app" as text
	set sheepPath to (POSIX path of sheepApp)
	try
		set macSystem to (path to me) & "Contents:Resources:Files:MacOS.sheepvm" as text
		set macPath to (POSIX path of macSystem)
	on error
		tell me to activate
		set dialogText to "The emulator package is missing from the application bundle. It should be in the folder Contents:Resources:Files."
		errorQuit(dialogText)
	end try
end setupVM

on runOK()
	set appRunning to false
	tell application "System Events"
		if (process "SheepShaver") exists then
			set appRunning to true
		end if
	end tell
	if appRunning is true then
		activate -- makes sure the dialog is not hidden
		display dialog "You cannot use the WPMac-based feature of the Converter application while SheepShaver running." buttons {"If you say so."} default button 1 with icon stop
		error number -128
	end if
	-- set appRunning to false
	tell application "System Events"
		set apps to every process whose bundle identifier is "org.wpdos.wpmacconverter-sheepshaver"
		if not (apps is {}) then
			set appRunning to true
		end if
	end tell
	if appRunning is true then
		activate -- makes sure the dialog is not hidden
		display dialog "You cannot use the WPMac-based feature of the Converter application while it is already running." buttons {"If you say so."} default button 1 with icon stop
		error number -128
	end if
end runOK

on getHelp()
	set thisApp to (path to me) as text
	set readmeFile to (thisApp & "Contents:Resources:Files:WP Converter ReadMe.rtf") as text
	set readmePosix to quoted form of (POSIX path of (readmeFile as alias))
	activate -- makes sure the dialog is not hidden
	set showHelp to display dialog "The WP Converter application exports WordPerfect documents to  RTF, DOC, or PDF format." & return & return & "The original WordPerfect file is not changed." & return & "I make a copy of the original file and convert the copy to the selected export format." & return & return & "I default to exporting to RTF format." & return & return & "If you change my name so that it includes DOC," & return & "I will export to the DOC format used by Microsoft Word for DOS 6.0." & return & return & "If you change my name so that it includes PDF," & return & "I will export to PDF format." & return & return & "You may drop one or more files on me, or launch me and select a file." & return & return & "Choose  Continue if you want me to convert a WordPerfect document. Choose Exit to cancel." & return & return & "Remove Help from my name to disable this message." buttons {"Exit", "Continue", "Open ReadMe File"} default button 2 with icon note
	if button returned of showHelp is "Open ReadMe File" then
		tell application "TextEdit"
			activate
			open file readmeFile as alias
		end tell
		error number -128
	else if button returned of showHelp is "Exit" then
		error number -128
	end if
end getHelp

on readFile(checkItem) -- by Stefan K. at macscripter.net
	try -- in case the whole file is empty
		set paragraphList to paragraphs of (read checkItem)
		repeat with i from 1 to 10
			set firstPara to item i of paragraphList
			if firstPara is not "" then return firstPara
		end repeat
	end try
	return ""
end readFile

on setVars()
	set noMac to false
	set allMac to false
	set doPDF to false
	set doWP3 to false
	set doWP5 to false
	set doWP6 to false
	set noStart to false
	set noQuit to false
	set useBII to false
	set useMac to false
	set macType to false
	set outUpper to ""
	set rejectedList to ""
	set errorList to ""
	set reportRejects to ""
	set reportErrors to ""
	set time_stamp to ""
end setVars

on errorQuit(dialogText) -- Fatal error, quitting script
	tell me to activate
	display dialog dialogText buttons {"OK"} default button 1 with icon stop with title "WP Converter" giving up after 5
	error number -128
end errorQuit

